#!/usr/bin/env python3
# -*- coding: utf-8 -*-

'''
Programmname:       Alkanbenenner
Autor:              Joachim Jakob (Kronberg-Gymnasium Aschaffenburg)
Einsatzgebiete:     - Homologe Reihe der Alkane
                      (Namen, Summenformeln, Siedepunkte)
                    - Benennung unverzweigter und verzweigter Alkane
                    - Konstitutions- bzw. Gerüstisomerie
Begleitmaterial:    Hefteintrag Kohlenwasserstoffe der 10. Jgst Chemie
                    Kapitel 1.2 Verzweigte Alkane
                    (vgl. Farbauswahl bei der Nomenklatur)

Copyright (C) 2011  Joachim Jakob

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
'''

import os
from tkinter import *
from tkinter import font
import string

# Schriftart und Schriftgröße für *Entry festlegen (vgl. auch vorletzte Zeile)
schriftart='dejavu'         # Alternativen: 'Liberation Sans' 'Bitstream Vera Sans'
                            #               'Arial' 'Helvetica' 'Nimbus Sans L' 'Sans'
schriftgroesse='15'

class Fenster:
    def __init__(self, master):
        frame=Frame(master)
        self.frame1=Frame(frame, pady=4, padx=4)
        self.frame2=Frame(frame, pady=4, padx=4)
        self.frame3=Frame(frame, pady=4, padx=4, bd=2, relief=RIDGE)
        self.frame4=Frame(frame, pady=4, padx=4)
        self.frame5=Frame(frame, pady=4, padx=4)
        self.frame6=Frame(frame, pady=4, padx=4)
        for f, row, column, columnspan, sticky in (
                (frame,       0, 0, 1, E+W),
                (self.frame1, 0, 0, 1, E+W),
                (self.frame2, 1, 0, 1, E+W),
                (self.frame3, 2, 0, 1, E+W),
                (self.frame4, 3, 0, 1, E+W),
                (self.frame5, 4, 0, 1, E+W),
                (self.frame6, 5, 0, 1, E+W)):
            f.grid(row=row, column=column, columnspan=columnspan, sticky=sticky)

        # Werte
        self.alkylreste={
        '1':'meth',
        '2':'eth',
        '3':'prop',
        '4':'but',
        '5':'pent',
        '6':'hex',
        '7':'hept',
        '8':'oct',
        '9':'non',
        '10':'dec',
        '11':'undec',
        '12':'dodec'
        }
        self.vorsilben_start={
        '1':'',             # Mono wird weggelassen
        '2':'Di',
        '3':'Tri',
        '4':'Tetra'
        }
        self.vorsilben_mitte={
        '1':'',             # Mono wird weggelassen
        '2':'di',
        '3':'tri',
        '4':'tetra'
        }
        # Name:('Summenformel', 'Siedetemperatur', 'Zahl der Isomere')
        #       ggfs. Abbildung, Schmelzpunkt, Molare Masse
        self.namen={
        ## unverzweigt
        'Methan':('CH₄',                                '-161','1'),
        'Ethan':('C₂H₆',                                '-88', '1'),
        'Propan':('C₃H₈',                               '-42', '1'),
        'Butan':('C₄H₁₀',                               '-1', '2'),
        'Pentan':('C₅H₁₂',                              '36', '3'),
        'Hexan':('C₆H₁₄',                               '68', '5'),
        'Heptan':('C₇H₁₆',                              '98', '9'),
        'Octan':('C₈H₁₈',                               '126', '18'),
        'Nonan':('C₉H₂₀',                               '150', '35'),
        'Decan':('C₁₀H₂₂',                              '174', '75'),
        'Undecan':('C₁₁H₂₄',                            '195', '159'),
        'Dodecan':('C₁₂H₂₆',                            '216', '355'),
        ## verzweigt
        # C3-Kette
        # eine Methylgruppe
        '2-Methylpropan':('C₄H₁₀',                      '-12', '2'), # vgl. "Isobutan"
        # zwei Methylgruppen
        '2,2-Dimethylpropan':('C₅H₁₂',                  '10', '3'),
        # C4-Kette
        # eine Methylgruppe
        '2-Methylbutan':('C₅H₁₂',                       '28', '3'),
        # zwei Methylgruppen
        '2,2-Dimethylbutan':('C₆H₁₄',                   '50', '5'),
        '2,3-Dimethylbutan':('C₆H₁₄',                   '58', '5'),
        # drei Methylgruppen
        '2,2,3-Trimethylbutan':('C₇H₁₆',                '81', '9'),
        # vier Methylgruppen
        '2,2,3,3-Tetramethylbutan':('C₈H₁₈',            '106', '18'),
        # C5 Kette
        # eine Methylgruppe
        '2-Methylpentan':('C₆H₁₄',                      '62' , '5'),
        '3-Methylpentan':('C₆H₁₄',                      '63' , '5'),
        # zwei Methylgruppen
        '2,2-Dimethylpentan':('C₇H₁₆',                  '78' , '9'),
        '2,3-Dimethylpentan':('C₇H₁₆',                  '89' , '9'),
        '3,3-Dimethylpentan':('C₇H₁₆',                  '86' , '9'),
        '2,4-Dimethylpentan':('C₇H₁₆',                  '80' , '9'),
        # drei Methylgruppen
        '2,2,3-Trimethylpentan':('C₈H₁₈',               '110' , '18'),
        '2,3,3-Trimethylpentan':('C₈H₁₈',               '115' , '18'),
        '2,2,4-Trimethylpentan':('C₈H₁₈',                '99' , '18'),  # vgl. "Isooctan"
        '2,3,4-Trimethylpentan':('C₈H₁₈',               '113' , '18'),
        # vier Methylgruppen
        '2,2,4,4-Tetramethylpentan':('C₉H₂₀',           '122', '35' ),
        '2,2,3,4-Tetramethylpentan':('C₉H₂₀',           'k.a.', '35'),
        '2,3,3,4-Tetramethylpentan':('C₉H₂₀',           'k.a.', '35'),
        '2,2,3,3-Tetramethylpentan':('C₉H₂₀',           'k.a.', '35'),
        # eine Ethylgruppe
        '3-Ethylpentan':('C₇H₁₆',                       '93', '9'),
        # zwei Ethylgruppen
        '3,3-Diethylpentan':('C₉H₂₀',                   '146', '35'),
        # Methylgruppen und Ethylgruppen gemischt
        # eine Ethylgruppe, eine Methylgruppe
        '3-Ethyl-3-methylpentan':('C₈H₁₈',              '118' , '18'),
        '3-Ethyl-2-methylpentan':('C₈H₁₈',              '115' , '18'),
        # eine Ethylgruppe, zwei Methylgruppen
        '3-Ethyl-2,3-dimethylpentan':('C₉H₂₀',          'k. A.', '35'),
        '3-Ethyl-2,2-dimethylpentan':('C₉H₂₀',          'k. A.', '35'),
        # eine Ethylgruppe, drei Methylgruppen
        '3-Ethyl-2,2,3-trimethylpentan':('C₁₀H₂₂',      'k. A.', '75'),
        '3-Ethyl-2,2,4-trimethylpentan':('C₁₀H₂₂',      'k. A.', '75'),
        # eine Ethylgruppe, vier Methylgruppen
        '3-Ethyl-2,2,3,4-tetramethylpentan':('C₁₁H₂₄',  'k. A.', '159'),
        '3-Ethyl-2,2,4,4-tetramethylpentan':('C₁₁H₂₄',  'k. A.', '159'),
        # zwei Ethylgruppen, eine Methylgruppe
        '3,3-Diethyl-2-methylpentan':('C₁₀H₂₂',         'k. A.', '75'),
        # zwei Ethylgruppen, zwei Methylgruppen
        '3,3-Diethyl-2,2-dimethylpentan':('C₁₁H₂₄',     'k. A.', '159'),
        '3,3-Diethyl-2,4-dimethylpentan':('C₁₁H₂₄',     'k. A.', '159'),
        # zwei Ethylgruppen, drei Methylgruppen
        '3,3-Diethyl-2,2,4-trimethylpentan':('C₁₂H₂₆',  'k.a.', '355'),
        # zwei Ethylgruppen, vier Methylgruppen
        '3,3-Diethyl-2,2,4,4-tetramethylpentan':('C₁₃H₂₈', 'k.a.', '802'),
        }

        # Rahmen 1: Evtl. Abbildung eines Moleküls => Nebenfenster

        # Rahmen 2: Eingabe der Werte, die aus der vorgegebenen Struktur abgelesen werden
        # Hauptkette
        self.f2l1=Label(self.frame2, text='Zahl der C-Atome \nin der Längsten Kette:', fg='darkgreen')
        self.f2scale1=Scale(self.frame2, length=400, from_=1, to=12, resolution=1,
                                         showvalue=1, orient=HORIZONTAL, fg='darkgreen',
                                         command=self.seitenkettenoptionen_festlegen)
        # Unverzweigt/verzweigt?
        global a1
        a1=StringVar()
        a1.set('unverzweigt')
        self.f2r1=Radiobutton(self.frame2,
                             variable=a1,
                             value='unverzweigt',
                             text='Unverzweigt',
                             command=self.auswahl_verzweigung)
        self.f2r2=Radiobutton(self.frame2,
                             variable=a1,
                             value='verzweigt',
                             text='Verzweigt',
                             command=self.auswahl_verzweigung)

        # Rahmen 3: Seitenkettentabelle
        self.f3l1=Label(self.frame3, text='Zahl der C-Atome:', fg='darkorange')
        self.f3l2=Label(self.frame3, text='Anzahl:')
        self.f3l3=Label(self.frame3, text='Positionen \n(aufsteigend):', fg='blue')
        # Typ 1
        self.f3l11=Label(self.frame3, text='Seitenkette:', fg='darkorange')
        self.f3l12=Label(self.frame3, text='1', fg='darkorange')
        self.f3scale13=Scale(self.frame3, length=120, from_=0, to=4, resolution=1,
                             showvalue=1, orient=HORIZONTAL,
                             command=self.aktiviere_entry1)
        self.f3e14=Entry(self.frame3, width='2', bg='white', fg='blue')
        self.f3e15=Entry(self.frame3, width='2', bg='white', fg='blue')
        self.f3e16=Entry(self.frame3, width='2', bg='white', fg='blue')
        self.f3e17=Entry(self.frame3, width='2', bg='white', fg='blue')
        # Typ 2
        self.f3l21=Label(self.frame3, text='Seitenkette:', fg='darkorange')
        self.f3l22=Label(self.frame3, text='2', fg='darkorange')
        self.f3scale23=Scale(self.frame3, length=120, from_=0, to=2, resolution=1,
                             showvalue=1, orient=HORIZONTAL,
                             command=self.aktiviere_entry2)
        self.f3e24=Entry(self.frame3, width='2', bg='white', fg='blue')
        self.f3e25=Entry(self.frame3, width='2', bg='white', fg='blue')

        # Rahmen 4: Ausgabe des Namens
        # Ethylrest Stellung, Anzahl, Name
        self.f5e11=Entry(self.frame4, width=4, fg='blue')
        self.f5l12=Label(self.frame4, text='-')
        self.f5e13=Entry(self.frame4, width=6)
        self.f5e14=Entry(self.frame4, width=6, fg='darkorange')
        # Methylrest Stellung, Anzahl, Name
        self.f5e21=Entry(self.frame4, width=6, fg='blue')
        self.f5l22=Label(self.frame4, text='-')
        self.f5e23=Entry(self.frame4, width=6)
        self.f5e24=Entry(self.frame4, width=6, fg='darkorange')
        # Hauptkette
        self.f5e31=Entry(self.frame4, width=6, fg='darkgreen')
        self.f5l32=Label(self.frame4, text='an', anchor=W, justify=LEFT)
        # Name komplett
        self.f5l33a=Label(self.frame4, text='Name:', anchor=E, justify=RIGHT)
        self.f5e33=Entry(self.frame4, width='30', justify=RIGHT)
        # Summenformel
        self.f5l34a=Label(self.frame4, text='Summenformel:', anchor=E, justify=RIGHT)
        self.f5e34=Entry(self.frame4, width='30', justify=RIGHT)
        # Siedepunkt und Anzahl der verschiedenen Isomere
        self.f5l35a=Label(self.frame4, text='Siedepunkt:', anchor=E, justify=RIGHT)
        self.f5e35=Entry(self.frame4, width='30', justify=RIGHT)
        self.f5l35b=Label(self.frame4, text='°C', anchor=W, justify=LEFT)
        self.f5l36a=Label(self.frame4, text='Anzahl der Gerüstisomere:', anchor=E, justify=RIGHT)
        self.f5e36=Entry(self.frame4, width='30', justify=RIGHT)


        # Rahmen 6: Navigationsknöpfe
        self.f6b0=Button(self.frame6, text='Name ermitteln', command=self.name_ermitteln)
        self.f6b1=Button(self.frame6, text='C-Kette anzeigen', command=self.strukturformel_anzeigen)
        self.f6b2=Button(self.frame6, text='Zurücksetzen', command=self.zurueck_setzen)
        self.f6b3=Button(self.frame6, text='Beenden', command=root.destroy)

        for i, row, columnspan, column, sticky in (
                # Rahmen 2
                (self.f2l1,                 0, 1, 0, E+W),
                (self.f2scale1,             0, 4, 1, E+W),
                (self.f2r1,                 1, 1, 0, E+W),
                (self.f2r2,                 1, 1, 1, E+W),
                # Rahmen 3
                (self.f3l1,                 2, 1, 1, E+W),
                (self.f3l2,                 2, 1, 2, E+W),
                (self.f3l3,                 2, 4, 3, E+W),
                (self.f3l11,                3, 1, 0, E+W),
                (self.f3l12,                3, 1, 1, E+W),
                (self.f3scale13,            3, 1, 2, E+W),
                (self.f3e14,                3, 1, 3, E+W),
                (self.f3e15,                3, 1, 4, E+W),
                (self.f3e16,                3, 1, 5, E+W),
                (self.f3e17,                3, 1, 6, E+W),
                (self.f3l21,                4, 1, 0, E+W),
                (self.f3l22,                4, 1, 1, E+W),
                (self.f3scale23,            4, 1, 2, E+W),
                (self.f3e24,                4, 1, 3, E+W),
                (self.f3e25,                4, 1, 4, E+W),
                # Rahmen 4
                (self.f5e11,                0, 1, 0, E+W),
                (self.f5l12,                0, 1, 1, E+W),
                (self.f5e13,                0, 1, 2, E+W),
                (self.f5e14,                0, 1, 3, E+W),
                (self.f5e21,                0, 1, 4, E+W),
                (self.f5l22,                0, 1, 5, E+W),
                (self.f5e23,                0, 1, 6, E+W),
                (self.f5e24,                0, 1, 7, E+W),
                (self.f5e31,                0, 1, 8, E+W),
                (self.f5l32,                0, 1, 10, E+W),
                (self.f5l33a,               1, 3, 0, E),
                (self.f5e33,                1, 7, 3, W),
                (self.f5l34a,               2, 3, 0, E),
                (self.f5e34,                2, 7, 3, W),
                (self.f5l35a,               3, 3, 0, E),
                (self.f5e35,                3, 7, 3, W),
                (self.f5l35b,               3, 1, 10, E),
                (self.f5l36a,               4, 3, 0, E),
                (self.f5e36,                4, 7, 3, W),
                # Rahmen 6
                (self.f6b0,                 0, 1, 0, E+W),
                (self.f6b1,                 0, 1, 1, E+W),
                (self.f6b2,                 0, 1, 2, E+W),
                (self.f6b3,                 0, 1, 3, E+W)):
            i.grid(row=row, columnspan=columnspan, column=column)
        self.auswahl_verzweigung()

        # Nebenfenster
        global unterFensterZahl
        unterFensterZahl = int(0)
        global namensLaengeUnterfenster
        namensLaengeUnterfenster = str('')

    def zurueck_setzen(self):
        #print('zurueck_setzen')
        a1.set('unverzweigt')
        self.auswahl_verzweigung()
        self.f2scale1.set(1)

    def seitenkettenoptionen_festlegen(self, a):
        #print('seitenkettenoptionen_festlegen')
        self.auswahl=str(a1.get())
        if self.auswahl == 'unverzweigt':
            self.name_ermitteln()
            self.f6b1.config(state=NORMAL)
        elif self.auswahl == 'verzweigt':
            self.kettenlaenge=str(self.f2scale1.get())
            #print self.kettenlaenge
            if self.kettenlaenge == '3':
                self.f3scale13.config(from_=0, to=2)
                self.f3scale13.set(0)
                self.f3scale23.set(0)
                self.f3scale23.config(state=DISABLED)
            elif self.kettenlaenge == '4':
                self.f3scale13.config(from_=0, to=4)
                self.f3scale13.set(0)
                self.f3scale23.set(0)
                self.f3scale23.config(state=DISABLED)
            elif self.kettenlaenge == '5':
                self.f3scale13.config(from_=0, to=4)
                self.f3scale13.set(0)
                self.f3scale23.config(state=NORMAL)
                self.f3scale23.set(0)
            # Hauptkette schon mal im Voraus anzeigen
            self.schluessel_kettenlaenge=str(self.f2scale1.get())
            self.stammname=self.alkylreste[self.schluessel_kettenlaenge]
            self.f5e31.config(state=NORMAL)
            self.f5e31.delete(0, END)
            self.f5e31.insert(0, self.stammname)
            self.f6b1.config(state=DISABLED)

    def aktiviere_entry1(self, a):
        self.anzahl=self.f3scale13.get()
        if  self.anzahl == 0:
            for entry in (self.f3e14, self.f3e15, self.f3e16, self.f3e17):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=DISABLED)
        elif self.anzahl == 1:
            for entry, state in ((self.f3e14, NORMAL),
                                 (self.f3e15, DISABLED),
                                 (self.f3e16, DISABLED),
                                 (self.f3e17, DISABLED)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)
        elif self.anzahl == 2:
            for entry, state in ((self.f3e14, NORMAL),
                                 (self.f3e15, NORMAL),
                                 (self.f3e16, DISABLED),
                                 (self.f3e17, DISABLED)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)
        elif self.anzahl == 3:
            for entry, state in ((self.f3e14, NORMAL),
                                 (self.f3e15, NORMAL),
                                 (self.f3e16, NORMAL),
                                 (self.f3e17, DISABLED)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)
        elif self.anzahl == 4:
            for entry, state in ((self.f3e14, NORMAL),
                                 (self.f3e15, NORMAL),
                                 (self.f3e16, NORMAL),
                                 (self.f3e17, NORMAL)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)


    def aktiviere_entry2(self, a):
        self.anzahl=self.f3scale23.get()
        if  self.anzahl == 0:
            for entry in (self.f3e24, self.f3e25):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=DISABLED)
        elif self.anzahl == 1:
            for entry, state in ((self.f3e24, NORMAL),
                                 (self.f3e25, DISABLED)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)
        elif self.anzahl == 2:
            for entry, state in ((self.f3e24, NORMAL),
                                 (self.f3e25, NORMAL)):
                entry.config(state=NORMAL)
                entry.delete(0, END)
                entry.config(state=state)

    def auswahl_verzweigung(self):
        #print('auswahl_verzweigung')
        self.auswahl=str(a1.get())
        if self.auswahl == 'unverzweigt':
            self.f2scale1.config(length=400, from_=1, to=12)
            self.f2scale1.set(1)
            for i in (self.f3scale13,
                      self.f3scale23):
                i.config(from_=0)
                i.set(0)
                i.config(state=DISABLED)
            for i in (self.f3e14, self.f3e15, self.f3e16, self.f3e17,
                      self.f3e24, self.f3e25,
                      self.f5e11, self.f5e13, self.f5e14,
                      self.f5e21, self.f5e23, self.f5e24,
                      self.f5e31, self.f5e33, self.f5e34,
                      self.f5e35, self.f5e36):
                i.config(state=NORMAL)
                i.delete(0, END)
                i.config(state=DISABLED)
        elif self.auswahl == 'verzweigt':
            self.f2scale1.config(length=400, from_=3, to=5)
            self.f2scale1.set(3)
            for i in (self.f3scale13,
                      self.f3scale23):
                i.config(state=NORMAL)
            for i in (self.f3e14, self.f3e15, self.f3e16, self.f3e17,
                      self.f3e24, self.f3e25,
                      self.f5e11, self.f5e13, self.f5e14,
                      self.f5e21, self.f5e23, self.f5e24,
                      self.f5e31, self.f5e33, self.f5e34,
                      self.f5e35, self.f5e36):
                i.config(state=NORMAL)
                i.delete(0, END)
                i.config(state=DISABLED)
            for i in (self.f5e11, self.f5e13, self.f5e14,
                      self.f5e21, self.f5e23, self.f5e24,
                      self.f5e31, self.f5e33, self.f5e34,
                      self.f5e35, self.f5e36):
                i.config(state=DISABLED)

    def name_ermitteln(self):
        self.schluessel_kettenlaenge=str(self.f2scale1.get())
        for i in (self.f5e11, self.f5e13, self.f5e14,
                  self.f5e21, self.f5e23, self.f5e24,
                  self.f5e31, self.f5e33, self.f5e34,
                      self.f5e35, self.f5e36):
            i.config(state=NORMAL, bg='white')
            i.delete(0, END)
        self.stammname=self.alkylreste[self.schluessel_kettenlaenge]
        self.anzahl_ethyl=self.f3scale23.get()
        self.anzahl_methyl=self.f3scale13.get()
        if self.anzahl_ethyl == 0:
            for i in (self.f5e11, self.f5e13, self.f5e14):
                i.delete(0, END)
                i.config(state=DISABLED)
        elif self.anzahl_ethyl == 1:
            self.position_ethyl_1=str(self.f3e24.get())
            self.anzahl_ethyl_wort=self.vorsilben_start[str(self.anzahl_ethyl)]
            self.f5e11.insert(0, self.position_ethyl_1)
            self.f5e13.config(state=DISABLED)
            self.f5e14.insert(0, 'Ethyl')
        elif self.anzahl_ethyl == 2:
            self.position_ethyl_1=str(self.f3e24.get())
            self.position_ethyl_2=str(self.f3e25.get())
            self.position_ethyl=self.position_ethyl_1+','+self.position_ethyl_2
            self.anzahl_ethyl_wort=self.vorsilben_start[str(self.anzahl_ethyl)]
            self.f5e11.insert(0, self.position_ethyl)
            self.f5e13.insert(0, self.anzahl_ethyl_wort)
            self.f5e14.insert(0, 'ethyl')
        if self.anzahl_methyl == 0:
            for i in (self.f5e21, self.f5e23, self.f5e24):
                i.delete(0, END)
                i.config(state=DISABLED)
        elif self.anzahl_methyl == 1 and self.anzahl_ethyl == 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.f5e21.insert(0, self.position_methyl_1)
            self.f5e24.insert(0, 'Methyl')
            self.f5e23.config(state=DISABLED)
        elif self.anzahl_methyl == 1 and self.anzahl_ethyl > 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.anzahl_methyl_wort=self.vorsilben_mitte[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl_1)
            self.f5e24.insert(0, 'methyl')
            self.f5e23.config(state=DISABLED)
        elif self.anzahl_methyl == 2 and self.anzahl_ethyl == 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2
            self.anzahl_methyl_wort=self.vorsilben_start[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        elif self.anzahl_methyl == 2 and self.anzahl_ethyl > 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2
            self.anzahl_methyl_wort=self.vorsilben_mitte[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        elif self.anzahl_methyl == 3 and self.anzahl_ethyl == 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl_3=str(self.f3e16.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2+','+self.position_methyl_3
            self.anzahl_methyl_wort=self.vorsilben_start[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        elif self.anzahl_methyl == 3 and self.anzahl_ethyl > 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl_3=str(self.f3e16.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2+','+self.position_methyl_3
            self.anzahl_methyl_wort=self.vorsilben_mitte[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        elif self.anzahl_methyl == 4 and self.anzahl_ethyl == 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl_3=str(self.f3e16.get())
            self.position_methyl_4=str(self.f3e17.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2+','+self.position_methyl_3+','+self.position_methyl_4
            self.anzahl_methyl_wort=self.vorsilben_start[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        elif self.anzahl_methyl == 4 and self.anzahl_ethyl > 0:
            self.position_methyl_1=str(self.f3e14.get())
            self.position_methyl_2=str(self.f3e15.get())
            self.position_methyl_3=str(self.f3e16.get())
            self.position_methyl_4=str(self.f3e17.get())
            self.position_methyl=self.position_methyl_1+','+self.position_methyl_2+','+self.position_methyl_3+','+self.position_methyl_4
            self.anzahl_methyl_wort=self.vorsilben_mitte[str(self.anzahl_methyl)]
            self.f5e21.insert(0, self.position_methyl)
            self.f5e23.insert(0, self.anzahl_methyl_wort)
            self.f5e24.insert(0, 'methyl')
        self.f5e31.delete(0, END)
        self.auswahl=str(a1.get())
        if self.auswahl== 'unverzweigt':
            self.stammname_anfangsbuchstabe=self.stammname[:1].upper()
            self.stammname_rest=self.stammname[1:]
            self.stammname=self.stammname_anfangsbuchstabe+self.stammname_rest
        self.f5e31.insert(0, self.stammname)
        self.n1=str(self.f5e11.get())
        if len(self.n1) > 0:
            self.n1=self.n1+'-'
        self.n2=str(self.f5e13.get())
        self.n3=str(self.f5e14.get())
        self.n4=str(self.f5e21.get())
        if len(self.n4) > 0 and len(self.n3) > 0:
            self.n4='-'+self.n4+'-'
        elif len(self.n4) > 0 and len(self.n3) == 0:
            self.n4=self.n4+'-'
        self.n5=str(self.f5e23.get())
        self.n6=str(self.f5e24.get())
        self.n7=str(self.f5e31.get())
        self.name= "%s%s%s%s%s%s%s%s" % (self.n1, self.n2, self.n3,
                                       self.n4, self.n5, self.n6,
                                       self.n7, 'an')
        self.f5e33.delete(0, END)
        self.f5e33.insert(0, self.name)
        self.name_ueberpruefen(self.name)

    def name_ueberpruefen(self, name):
        #print('name_ueberpruefen')
        self.name=name
        if self.name in self.namen:
            #print("gibt's wirklich")
            self.f5e33.config(bg='white')
            self.f5e34.config(bg='white')
            self.f5e35.config(bg='white')
            self.f5e36.config(bg='white')
            self.summenformel=str(self.namen[self.name][0])
            self.siedepunkt=str(self.namen[self.name][1])
            self.isomere=str(self.namen[self.name][2])
            self.f5e34.insert(0, self.summenformel)
            self.f5e35.insert(0, self.siedepunkt)
            self.f5e36.insert(0, self.isomere)
            self.f6b1.config(state=NORMAL)
        else:
            #print("gibt's nicht")
            self.f5e33.config(bg='red')
            self.f5e33.config(bg='red')
            self.f6b1.config(state=DISABLED)
        # TO DO: Gegebenenfalls Fehlermeldung anzeigen oder
        #        richtige Strukturformel generieren/anzeigen

    def strukturformel_anzeigen(self):
        print('strukturformel_anzeigen')
        self.strukturformel_fenster_test()

        ########################################################################
        ##                          Nebenfenster                              ##
        ########################################################################

        global unterFensterZahl
        unterFensterZahl = int(0)
        global namensLaengeUnterfenster
        namensLaengeUnterfenster = str('')

    def strukturformel_fenster_test(self):
        global namensLaengeUnterfenster
        global unterFensterZahl
        try:
            namensLaengeUnterfenster = self.nw.winfo_name()
            if len(namensLaengeUnterfenster) > 1:
                self.nw.destroy()
        except:
            pass
            unterFensterZahl = 0
        if unterFensterZahl < 1:
            self.strukturformel_fenster()

    def strukturformel_fenster(self):
        self.nw = Toplevel(root)
        # Zahl der C-Atome in der Hauptkette
        self.hauptkettenlaenge=int(self.f2scale1.get())
        # Positionen 1 bis 4 der Methyl-Seitenketten
        self.methyl_1=self.f3e14.get()
        self.methyl_2=self.f3e15.get()
        self.methyl_3=self.f3e16.get()
        self.methyl_4=self.f3e17.get()
        # Positionen 1 bis 2 der Ethyl-Seitenketten
        self.ethyl_1=self.f3e24.get()
        self.ethyl_2=self.f3e25.get()
        print(self.hauptkettenlaenge,
              self.methyl_1,
              self.methyl_2,
              self.methyl_3,
              self.methyl_4,
              self.ethyl_1,
              self.ethyl_2)
        self.molekuelname=self.f5e33.get()
        self.nw.title('C-Kette des '+self.molekuelname+' Moleküls')
        self.nwFrame=Frame(self.nw)
        self.nwFrame.grid(row=0, column=0)
        self.nwCanvas=Canvas(self.nwFrame, width=600, height=400, bg='white')
        self.nwCanvas.grid(row=0, column=0)
        self.nwButton=Button(self.nwFrame, text='Fenster Schließen',
                         command=self.strukturformel_fenster_schliessen)
        self.nwButton.grid(row=1, column=0)
        # Evtl. noch vorhandene Elemente löschen
        for i in self.nwCanvas.find_all():
                self.nwCanvas.delete(i)
        # C-Hauptkette aus den Klassenbausteinen generieren und anzeigen
        if self.hauptkettenlaenge == 1:
            self.hauptkette=methan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 2:
            self.hauptkette=ethan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 3:
            self.hauptkette=propan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 4:
            self.hauptkette=butan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 5:
            self.hauptkette=pentan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 6:
            self.hauptkette=hexan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 7:
            self.hauptkette=heptan_kette(self.nwCanvas, 50, 200)
        elif self.hauptkettenlaenge == 8:
            self.hauptkette=octan_kette(self.nwCanvas, 50, 200)
            self.nwCanvas.config(width=640)
        elif self.hauptkettenlaenge == 9:
            self.hauptkette=nonan_kette(self.nwCanvas, 50, 200)
            self.nwCanvas.config(width=660)
        elif self.hauptkettenlaenge == 10:
            self.hauptkette=decan_kette(self.nwCanvas, 50, 200)
            self.nwCanvas.config(width=700)
        elif self.hauptkettenlaenge == 11:
            self.hauptkette=undecan_kette(self.nwCanvas, 50, 200)
            self.nwCanvas.config(width=740)
        elif self.hauptkettenlaenge == 12:
            self.hauptkette=dodecan_kette(self.nwCanvas, 50, 200)
            self.nwCanvas.config(width=780)
        # Positionen:
        # x-Werte:
        self.x_c2=110
        self.x_c3=170
        self.x_c4=230
        # y-Werte:
        self.y_oben_methyl=140
        self.y_unten_methyl=235
        self.y_oben_ethyl=70
        self.y_unten_ethyl=235

        # Seitenketten
        self.auswahl=str(a1.get())
        if self.auswahl == 'verzweigt':
            if self.hauptkettenlaenge == 3 or self.hauptkettenlaenge == 4:
                #print('hauptkettenlaenge = 3 oder 4')
                if self.methyl_1 == '2':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2, self.y_oben_methyl)
                    #print('self.methyl_1_kette=methyl_von_oben')
                if self.methyl_2 == '2':
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2, self.y_unten_methyl)
                if self.methyl_2 == '3' and self.ethyl_1 == '':
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c3, self.y_oben_methyl)
                if self.methyl_3 == '3' and self.ethyl_1 == '':
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c3, self.y_oben_methyl)
                if self.methyl_4 == '3' and self.ethyl_1 == '':
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c3, self.y_unten_methyl)
            elif self.hauptkettenlaenge == 5:
                #print('hauptkettenlaenge = 5')
                if self.methyl_1 == '':
                    self.methyl_1=self.methyl_1+'X'
                if self.methyl_2 == '':
                    self.methyl_2=self.methyl_2+'X'
                if self.methyl_3 == '':
                    self.methyl_3=self.methyl_3+'X'
                if self.methyl_4 == '':
                    self.methyl_4=self.methyl_4+'X'
                if self.ethyl_1 == '':
                    self.ethyl_1=self.ethyl_1+'X'
                if self.ethyl_2 == '':
                    self.ethyl_2=self.ethyl_2+'X'
                self.schluessel=self.methyl_1+self.methyl_2+self.methyl_3+self.methyl_4+self.ethyl_1+self.ethyl_2
                #print(self.schluessel)
                if self.schluessel == '2XXXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                elif self.schluessel == '3XXXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                elif self.schluessel == '22XXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                elif self.schluessel == '23XXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                elif self.schluessel == '33XXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '24XXXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '223XXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                elif self.schluessel == '233XXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_3_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '224XXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '234XXX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '2244XX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c4,  self.y_unten_methyl)
                elif self.schluessel == '2234XX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '2334XX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_3_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                    self.methyl_4_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '2233XX':
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c3,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == 'XXXX3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                elif self.schluessel == 'XXXX33':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                elif self.schluessel == '3XXX3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '2XXX3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                elif self.schluessel == '23XX3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '22XX3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                elif self.schluessel ==  '223X3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '224X3X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '22343X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c3,  self.y_unten_methyl)
                elif self.schluessel == '22443X':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c4,  self.y_unten_methyl)
                elif self.schluessel == 'XXXX33':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                elif self.schluessel == '22XX33':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                elif self.schluessel == '24XX33':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '224X33':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                elif self.schluessel == '224433':
                    self.ethyl_1_kette=ethyl_von_oben(self.nwCanvas,        self.x_c3,  self.y_oben_ethyl)
                    self.ethyl_2_kette=ethyl_von_unten(self.nwCanvas,       self.x_c3,  self.y_unten_ethyl)
                    self.methyl_1_kette=methyl_von_oben(self.nwCanvas,      self.x_c2,  self.y_oben_methyl)
                    self.methyl_2_kette=methyl_von_unten(self.nwCanvas,     self.x_c2,  self.y_unten_methyl)
                    self.methyl_3_kette=methyl_von_oben(self.nwCanvas,      self.x_c4,  self.y_oben_methyl)
                    self.methyl_4_kette=methyl_von_unten(self.nwCanvas,     self.x_c4,  self.y_unten_methyl)

    def strukturformel_fenster_schliessen(self):
        self.nw.destroy()
        global unterFensterZahl
        if unterFensterZahl >= 1:
            try:
                unterFensterZahl = 0
            except:
                pass

# unverzweigt
class methan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class ethan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class propan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class butan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class pentan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class hexan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class heptan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class octan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class nonan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class decan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class undecan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

class dodecan_kette:
    def __init__(self, canvas, x1, y1):
        # C
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C ― C ― C ― C ― C ― C ― C ― C ― C ― C ― C ― C', fill='darkgreen', font=('helvetica', 20), anchor=NW, justify=LEFT)

# verzweigt
class methyl_von_oben:
    def __init__(self, canvas, x1, y1):
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        # C
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # |
        self.bd1=self.canvas.create_text(self.x1+5, self.y1+30, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)

class methyl_von_unten:
    def __init__(self, canvas, x1, y1):
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        # |
        self.bd1=self.canvas.create_text(self.x1+5, self.y1, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # C
        self.c1=self.canvas.create_text(self.x1, self.y1+30, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)

class ethyl_von_oben:
    def __init__(self, canvas, x1, y1):
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        # C
        self.c1=self.canvas.create_text(self.x1, self.y1, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # |
        self.bd1=self.canvas.create_text(self.x1+5, self.y1+30, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # C
        self.c2=self.canvas.create_text(self.x1, self.y1+60, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # |
        self.bd2=self.canvas.create_text(self.x1+5, self.y1+90, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)

class ethyl_von_unten:
    def __init__(self, canvas, x1, y1):
        self.canvas=canvas
        self.x1=x1
        self.y1=y1
        # |
        self.c1=self.canvas.create_text(self.x1+5, self.y1, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # C
        self.bd1=self.canvas.create_text(self.x1, self.y1+30, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # |
        self.c2=self.canvas.create_text(self.x1+5, self.y1+60, text='|', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
        # C
        self.bd2=self.canvas.create_text(self.x1, self.y1+90, text='C', fill='darkorange', font=('helvetica', 20), anchor=NW, justify=LEFT)
#############################################################################
root = Tk()
root.title('Alkanbenenner')
root.option_add("*Entry.Font", schriftart+' '+schriftgroesse)
app = Fenster(root)
root.mainloop()
